package xmpp.cebutils;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import org.conscrypt.Conscrypt;
import org.jxmpp.stringprep.libidn.LibIdnXmppStringprep;


public class Main {

    public static final String KEYTYPE = "AES";
    public static final String CIPHERMODE = "AES/GCM/NoPadding";
    
    static {
        LibIdnXmppStringprep.setup();
    }

    public static void main(final String... args) throws Exception {
        if (args.length != 1 && args.length != 2) {
            System.err.println("usage:");
            System.err.println("  java -jar xmpp.cebutils.jar [source_ceb_file]");
            System.err.println("       : decrypt source_ceb_file to source_ceb_file.txt");
            System.err.println("");
            System.err.println("  java -jar xmpp.cebutils.jar [source_ceb_file] [txt_file]");
            System.err.println("       : encrypt txt_file using source_ceb_file parameters, to txt_file.ceb");
            System.exit(1);
        }
        final String cebFile = args[0];

        String txtFile = null;
        if (args.length == 2) {
             txtFile = args[1];
        }

	/* always read cebFile to make sure encryption validates */
        final File file = new File(cebFile);
        final FileInputStream fileInputStream = new FileInputStream(file);
        final DataInputStream dataInputStream = new DataInputStream(fileInputStream);
        final BackupFileHeader backupFileHeader;

        try {
            backupFileHeader = BackupFileHeader.read(dataInputStream);
        } catch (final Exception e) {
            System.err.println(file.getAbsolutePath() + " does not seem to be a valid backup file");
            System.exit(1);
            return;
        }
        final Console console = System.console();

        final String password =
                new String(
                        console.readPassword(
                                "Enter password for "
                                        + backupFileHeader.getJid().asBareJid()
                                        + ": "));

        final Cipher cipher = Cipher.getInstance(CIPHERMODE, Conscrypt.newProvider());
        byte[] key = getKey(password, backupFileHeader.getSalt());
        byte[] iv = backupFileHeader.getIv();
        byte[] salt = backupFileHeader.getSalt();
        System.out.println("salt, iv, [final] key:");
        for (byte b : salt) { System.out.printf("%02x",b); }
        System.out.printf("\n");
        for (byte b : iv) { System.out.printf("%02x",b); }
        System.out.printf("\n");
        for (byte b : key) { System.out.printf("%02x",b); }
        System.out.printf("\n");

        BufferedReader reader = null;

        final SecretKeySpec keySpec = new SecretKeySpec(key, KEYTYPE);
        final IvParameterSpec ivSpec = new IvParameterSpec(backupFileHeader.getIv());
        try {
            cipher.init(Cipher.DECRYPT_MODE, keySpec, ivSpec);
            final CipherInputStream cipherInputStream =
                    new CipherInputStream(fileInputStream, cipher);

            final GZIPInputStream gzipInputStream = new GZIPInputStream(cipherInputStream);
            reader =
                    new BufferedReader(
                            new InputStreamReader(gzipInputStream, StandardCharsets.UTF_8));
        } catch (InvalidAlgorithmParameterException e) {
            System.err.println("Correct backup file");
            System.exit(1);
            return;
        } catch (IOException e) {
            System.err.println("Wrong password or corrupt backup file");
            System.exit(1);
            return;
        }

        /* if there's a plaintext file supplied by the user,
         * output file is an encrypted ceb
         */
        OutputStreamWriter outputWriter = null;
        if (txtFile != null) {

            final FileInputStream nfileInputStream = new FileInputStream(new File(txtFile));
            final DataInputStream ndataInputStream = new DataInputStream(nfileInputStream);
            reader = new BufferedReader(new InputStreamReader(ndataInputStream, StandardCharsets.UTF_8));

            final File fixedFile = new File(".", txtFile + ".ceb");
            System.out.println("writing encrypted .ceb to " + txtFile + ".ceb");

            final OutputStream outputStream = new FileOutputStream(fixedFile);
            final DataOutputStream dataOutputStream = new DataOutputStream(outputStream);
            backupFileHeader.write(dataOutputStream);

            cipher.init(Cipher.ENCRYPT_MODE, keySpec, ivSpec);
            CipherOutputStream cipherOutputStream = new CipherOutputStream(dataOutputStream, cipher);

            final GZIPOutputStream gzipOutputStream = new GZIPOutputStream(cipherOutputStream);
            outputWriter = new OutputStreamWriter(gzipOutputStream, StandardCharsets.UTF_8);
        } else {
            /* output file is decrypted ceb plaintext */
            final File decFile = new File(".", cebFile + ".txt");
            System.out.println("writing decrypted .ceb to " + cebFile + ".txt");

            final OutputStream outputStream = new FileOutputStream(decFile);
            final DataOutputStream dataOutputStream = new DataOutputStream(outputStream);
            outputWriter = new OutputStreamWriter(dataOutputStream, StandardCharsets.UTF_8);
        }

        String l;
        while ((l=reader.readLine()) != null) {
            outputWriter.write(l + "\n");
        }

        outputWriter.flush();
        outputWriter.close();
    }

    public static byte[] getKey(final String password, final byte[] salt) {
        try {
            SecretKeyFactory factory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1");
            return factory.generateSecret(new PBEKeySpec(password.toCharArray(), salt, 1024, 128))
                    .getEncoded();
        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new AssertionError(e);
        }
    }

}
